using System;
using System.ComponentModel;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Xml.Serialization;
using System.Collections;
using Microsoft.SharePoint;
using Microsoft.SharePoint.Utilities;
using Microsoft.SharePoint.WebControls;
using Microsoft.SharePoint.WebPartPages;

namespace EasyTabParts
{
	/// <summary>
	/// PageBrowser - Web Part to allow browsing to pages within a WSS site or SPS area
	/// </summary>
	[DefaultProperty("ShowHome"),
		ToolboxData("<{0}:PageBrowser runat=server></{0}:PageBrowser>"),
		XmlRoot(Namespace="TestLib")]
	public class PageBrowser : EasyStringConsumer
	{
		#region ToolPart Attributes

		private const string defaultDocumentLibrary = "";
		private string documentLibrary = defaultDocumentLibrary;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultDocumentLibrary),
		WebPartStorage(Storage.Personal),
		FriendlyName("Document Library"),Description("Document Library containing pages")]
		public string DocumentLibrary
		{
			get
			{
				return documentLibrary;
			}

			set
			{
				documentLibrary = value;
			}
		}

		private const string defaultSortColumn = "ID";
		private string sortColumn = defaultSortColumn;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultSortColumn),
		WebPartStorage(Storage.Personal),
		FriendlyName("Sort Column"),Description("Column to Sort By")]
		public string SortColumn
		{
			get
			{
				return sortColumn;
			}

			set
			{
				sortColumn = value;
			}
		}

		private const bool defaultShowHome = false;
		private bool showHome = defaultShowHome;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultShowHome),
		WebPartStorage(Storage.Personal),
		FriendlyName("Show Home"),Description("Enables link to main site page")]
		public bool ShowHome
		{
			get
			{
				return showHome;
			}

			set
			{
				showHome = value;
			}
		}

		private const string defaultShowHomeTitle = "";
		private string showHomeTitle = defaultShowHomeTitle;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultShowHomeTitle),
		WebPartStorage(Storage.Personal),
		FriendlyName("Show Home Title"),Description("Title for link to main page")]
		public string ShowHomeTitle
		{
			get
			{
				return showHomeTitle;
			}

			set
			{
				showHomeTitle = value;
			}
		}

		public enum WPStyle
		{
			Links,
			Tabs
		}

		private const WPStyle defaultNavStyle = WPStyle.Tabs;
		private WPStyle navStyle = defaultNavStyle;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultNavStyle),
		WebPartStorage(Storage.Personal),
		FriendlyName("Web Part Style"),Description("Controls style for rendering")]
		public WPStyle NavStyle
		{
			get
			{
				return navStyle;
			}

			set
			{
				navStyle = value;
			}
		}

		public enum WPOrientation
		{
			Horizontal,
			Vertical
		}

		private const WPOrientation defaultNavOrientation = WPOrientation.Horizontal;
		private WPOrientation navOrientation = defaultNavOrientation;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultNavOrientation),
		WebPartStorage(Storage.Personal),
		FriendlyName("Web Part Orientation"), Description("Controls vertical or horizontal orientation")]
		public WPOrientation NavOrientation
		{
			get
			{
				return navOrientation;
			}

			set
			{
				navOrientation = value;
			}
		}

		private const string defaultQueryString = "";
		private string queryString = defaultQueryString;

		[Browsable(false),Category("Miscellaneous"),
		DefaultValue(defaultQueryString),
		WebPartStorage(Storage.Personal),
		FriendlyName("Query String"), Description("Query string to append to page references")]
		public string QueryString
		{
			get
			{
				return queryString;
			}

			set
			{
				queryString = value;
			}
		}

		#endregion

		#region ToolPart Registration
		/// <summary>
		///	Gets the custom tool parts for this Web Part by overriding the
		///	GetToolParts method of the WebPart base class. You must implement
		///	custom tool parts in a separate class that derives from 
		///	Microsoft.SharePoint.WebPartPages.ToolPart. 
		///	</summary>
		///<returns>An array of references to ToolPart objects.</returns>
		public override ToolPart[] GetToolParts()
		{
			ToolPart[] toolparts = new ToolPart[3];
			
			// Our own tool part
			PageBrowserToolPart pbtp = new PageBrowserToolPart();
			toolparts[0] = pbtp;

			// Base tool parts
			WebPartToolPart wptp = new WebPartToolPart();
			toolparts[1] = wptp;

			// Tool parts discovered by reflection
			CustomPropertyToolPart custom = new CustomPropertyToolPart();
			toolparts[2] = custom;

			return toolparts;
		}
		#endregion

		#region Attributes controlled by DWP only

		// LeaveEmptySpace - true if we should 1 tab width blank at the right
		private const bool defaultLeaveEmptySpace = true;
		private bool leaveEmptySpace = defaultLeaveEmptySpace;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultLeaveEmptySpace),
		WebPartStorage(Storage.Shared),
		FriendlyName("Leave Empty Space"),
		Description("Leaves empty space to the right of the last tab")]
		public bool LeaveEmptySpace
		{
			get
			{
				return leaveEmptySpace;
			}

			set
			{
				leaveEmptySpace = value;
			}
		}

		// SelectedTabCss - Holds the CSS class name for selected tabs
		private const string defaultSelectedTabCss = "ms-tabselected";
		private string selectedTabCss = defaultSelectedTabCss;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultSelectedTabCss),
		WebPartStorage(Storage.Shared),
		FriendlyName("Selected Tab CSS"),
		Description("CSS class for selected tabs")]
		public string SelectedTabCss
		{
			get
			{
				return selectedTabCss;
			}

			set
			{
				selectedTabCss = value;
			}
		}

		// UnselectedTabCss - Holds the CSS class name for selected tabs
		private const string defaultUnselectedTabCss = "ms-tabinactive";
		private string unselectedTabCss = defaultUnselectedTabCss;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultUnselectedTabCss),
		WebPartStorage(Storage.Shared),
		FriendlyName("Un-selected Tab CSS"),
		Description("CSS class for inactive tabs")]
		public string UnselectedTabCss
		{
			get
			{
				return unselectedTabCss;
			}

			set
			{
				unselectedTabCss = value;
			}
		}

		// TabBarCss - Holds the CSS class name bars above and below the tabs
		private const string defaultTabBarCss = "ms-tabselected";
		private string tabBarCss = defaultTabBarCss;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultTabBarCss),
		WebPartStorage(Storage.Shared),
		FriendlyName("Tab Bar CSS"),
		Description("CSS class bars above and below the tabs")]
		public string TabBarCss
		{
			get
			{
				return tabBarCss;
			}

			set
			{
				tabBarCss = value;
			}
		}

		#endregion

		#region Exception Handling
		// Web part exception handling
   		bool haveError = false;
		string errorMessage = "";
		private void HandleError (Exception ex)
		{
			haveError = true;
			errorMessage = ex.Message;
		}
		private bool HaveError (HtmlTextWriter output)
		{
			if (haveError)
			{
				output.Write(SPEncode.HtmlEncode(errorMessage));
				return (true);
			}
			return (false);
		}
		#endregion

		#region Web Part Rendering

		// Define the PageItem class, which will be used to allow us to sort the pages in an ArrayList
		private class PageItem : IComparable
		{
			public string BaseName;
			public string Url;
			public string SortKey;

			// IComparable Members

			public int CompareTo(object obj)
			{
				if (!(obj is PageItem))
				{
					throw new ArgumentException ("Attempt to compare PageItem with incompatible class");
				}
				PageItem pi = obj as PageItem;
				return (this.SortKey.CompareTo (pi.SortKey));
			}
		}

		// Utility method to build the ArrayList of pages, sorted by the specified column
		private ArrayList GetSortedPageItems (SPList pageLibrary)
		{
			ArrayList result = new ArrayList();

			foreach (SPListItem li in pageLibrary.Items)
			{
				PageItem pitem = new PageItem();

				SPFile pageFile = li.File;
				pitem.BaseName = li["BaseName"].ToString();
				pitem.Url = pageFile.Url;
				pitem.SortKey = li[this.SortColumn].ToString();

				result.Add (pitem);
			}

			result.Sort ();
			return (result);
		}

		// Table for rendering our page links
		Table pageTable;

		// Create the controls to render the table, populated with the page data
		protected override void CreateChildControls()
		{
			try
			{
				base.CreateChildControls ();

				// First, ensure we have a document library of pages. If not, throw an error which
				// will be displayed to the user
				if (documentLibrary == "")
				{
					throw new Exception ("Please modify this web part and select a document library " +
						" containing the pages you wish to navigate.");
				}

				// OK, we have a document library, go get it
				SPWeb thisWeb = SPControl.GetContextWeb (Context);
				string thisWebUrl = thisWeb.Url;
				thisWeb.Lists.IncludeRootFolder = true;
				SPList pageLibrary = thisWeb.Lists[documentLibrary];

				// This links the title bar to the document library
				this.DetailLink = thisWebUrl + "/" + pageLibrary.RootFolder.Url;

				// OK, now create the table which will hold the tabs, and declare variables to
				// reference controls within the table
				pageTable = new Table();

				int cellCount = 0;
				TableRow pageRow;
				TableCell pageCell;
				HyperLink pageLink;

				pageRow = new TableRow();

				// Since the "default.aspx" page is not in the page library, we can show it as
				// a tab if the showHome flag is set
				if (showHome)
				{
					pageCell = new TableCell();
					pageLink = new HyperLink();

					pageLink.Text = showHomeTitle;
					string navUrl = thisWebUrl + "/default.aspx";
					// Here we can pass context from a connected web part to the destination page
					if (queryString != null && queryString != "")
					{
						navUrl += "?" + queryString + "=%connectedValue%";
					}
					pageLink.NavigateUrl = navUrl;

					// Set the style attributes of the cell
					DecorateCell (pageCell, "default.aspx");

					pageCell.Controls.Add (pageLink);
					pageRow.Controls.Add (pageCell);
					cellCount++;

					// If we're vertical, we need a new row for the tab
					if (navOrientation == WPOrientation.Vertical)
					{
						pageTable.Controls.Add (pageRow);
					}
				}

				// Now get the list of pages that need tabs
				ArrayList sortedPageItems = GetSortedPageItems (pageLibrary);

				// Render each one as a tab
				foreach (Object o in sortedPageItems)
				{
					PageItem pitem = o as PageItem;
					if (pitem != null)
					{
						// If we're vertical then we need to start a new row
						if (navOrientation == WPOrientation.Vertical)
						{
							pageRow = new TableRow();
						}
						pageCell = new TableCell();
						pageLink = new HyperLink();

						pageLink.Text = pitem.BaseName;
						string navUrl = thisWebUrl + "/" + pitem.Url;
						if (queryString != null && queryString != "")
						{
							navUrl += "?" + queryString + "=%connectedValue%";
						}
						pageLink.NavigateUrl = navUrl;
					
						DecorateCell (pageCell, pitem.Url);

						pageCell.Controls.Add (pageLink);
						pageRow.Cells.Add (pageCell);
						cellCount++;
						// If we're vertical, then we need to add the new row to the table
						if (navOrientation == WPOrientation.Vertical)
						{
							pageTable.Controls.Add (pageRow);
						}
					}
				}

				// If we're horizontal then we're still on the same row as we started with,
				// so add it to the table
				if (navOrientation == WPOrientation.Horizontal)
				{
					pageTable.Controls.Add (pageRow);
				}

//				// If the user is an admin, place an "Add Page" link at the bottom
//				if (thisWeb.UserIsWebAdmin)
//				{
//					addPageUrl = thisWeb.Url				+ "/_layouts/1033/spcf.aspx?List="
//						+ System.Web.HttpUtility.UrlEncode (pageLibrary.ID.ToString())
//						+ "&RootFolder=&RootFolderUrl="
//						+ System.Web.HttpUtility.UrlEncode (thisWeb.Url + "/" + pageLibrary.RootFolder.Url);
//
//					pageRow = new TableRow();
//					pageCell = new TableCell();
//					if (navOrientation == WPOrientation.Horizontal)
//					{
//						pageCell.ColumnSpan = cellCount;
//					}
//					pageLink = new HyperLink();
//					
//					pageLink.Text = "(add page)";
//					pageLink.NavigateUrl = addPageUrl;
//
//					pageCell.Controls.Add (pageLink);
//					pageLink.Style.Add ("font-style", "italic");
//					pageCell.Style.Add ("text-align", "right");
//
//					pageRow.Cells.Add (pageCell);
//					pageTable.Rows.Add (pageRow);
//				}

				// OK, set the style for the whole table and add it to the web part
				DecorateTable (pageTable);

				this.Controls.Add (pageTable);
			}
			catch (Exception ex)
			{
				HandleError (ex);
			}
		}

		private void DecorateCell (TableCell cell, string pageUrl)
		{
			if (navStyle == WPStyle.Links)
			{
				cell.Style.Add ("display", "block");
				cell.Style.Add ("padding-left", "8px");
				if (this.Page.Request.Url.ToString().IndexOf (pageUrl) >= 0)
				{
					cell.Style.Add ("background-image", "url(/_layouts/images/CURVIEW.GIF)");
					cell.Style.Add ("background-repeat", "no-repeat");
					cell.Style.Add ("background-position", "left center");
				}
			}
			else
			{
				cell.Height = Unit.Pixel (25);
				if (navOrientation == WPOrientation.Horizontal)
				{
					cell.HorizontalAlign = HorizontalAlign.Center;
				}
				else
				{
					cell.HorizontalAlign = HorizontalAlign.Left;
				}
				if (this.Page.Request.Url.ToString().IndexOf (pageUrl) >= 0)
				{
					cell.CssClass = "ms-tabselected";
				}
				else
				{
					cell.CssClass = "ms-tabinactive";
				}
			}
		}

		private void DecorateTable (Table pageTable)
		{
			pageTable.Width = Unit.Percentage (100);

			if (navStyle == WPStyle.Tabs)
			{
				pageTable.CellSpacing = 3;
				pageTable.CellPadding = 3;
				if (navOrientation == WPOrientation.Horizontal)
				{
					double cellWidthPercent = 100.0 / pageTable.Rows[0].Cells.Count;
					foreach (TableCell cell in pageTable.Rows[0].Cells)
					{
						cell.Width = Unit.Percentage (cellWidthPercent);
					}
				}
			}
		}

		/// <summary>
		/// Render this Web Part to the output parameter specified.
		/// </summary>
		/// <param name="output"> The HTML writer to write out to </param>
		protected override void RenderWebPart(HtmlTextWriter output)
		{
			if (!HaveError (output))
			{
				try
				{
					string queryStringValue = this.GetInputString();
					if (queryStringValue == "" && queryString != "")
					{
						// We had no connected value, but a query string was provided in the tool part
						// So let's try to grab the value off the current page's query string
						queryStringValue = Page.Request.QueryString [queryString];
					}

					if (queryStringValue != null && queryStringValue != "")
					{
						foreach (TableRow tr in pageTable.Rows)
						{
							foreach (TableCell tc in tr.Cells)
							{
								HyperLink link = tc.Controls[0] as HyperLink;
								if (link != null)
								{
									string url = link.NavigateUrl;
									link.NavigateUrl = 
										url.Replace ("%connectedValue%", queryStringValue);
								}
							}
						}
					}
					else
					{
						foreach (TableRow tr in pageTable.Rows)
						{
							foreach (TableCell tc in tr.Cells)
							{
								HyperLink link = tc.Controls[0] as HyperLink;
								if (link != null)
								{
									string url = link.NavigateUrl;
									int i = url.IndexOf ("?");
									if (i>=0)
									{
										url = url.Substring (0, i);
										link.NavigateUrl = url;
									}
								}
							}
						}
					}
					base.RenderWebPart (output);
				}
				catch (Exception ex)
				{
					output.Write (ex.Message);
				}
			}
			else
			{
				output.Write (this.errorMessage);
			}
		}
		#endregion
	}
}
